/*
 * morph.c
 * Tommaso Polonelli
 *
 * Copyright (C) 2016 ETH Zurich, University of Bologna
 * Copyright (C) 2018 Tommaso Polonelli
 *
 * This software may be modified and distributed under the terms
 * of the MIT license.  See the LICENSE file for details.
 *
 * Created on: May 28, 2020
 *
 */

#include "equalize_hist.h"
#include "fixed_p.h"
#include <string.h>
#include "morph.h"

/*********** Constants  ****************/

#define PERF		(1)
#define PRINT_IMG	(0)

/*********** Structures *************/

/*********** Declarations *************/

/*------ Global variables  -----------*/

/*------ Function prototypes ---------*/


/*********** Functions   ****************/


m_kernel_t ConfigDefaultKernel(e_kernel_e e){

/*	K.rowOrigin and K.colOrigin must be in a 1 position
 * if the origin value is correspondent  to 0 the function wont work properly
 */
	m_kernel_t K;
	uint8_t *p;

	/* Init dimensions */
	switch (e){
	case K_STAR_3:
		K.Rows = 3;
		K.Cols = 3;
		K.rowOrigin = 1;
		K.colOrigin = 1;
		break;
	case K_CIRCLE_3:
		K.Rows = 3;
		K.Cols = 3;
		K.rowOrigin = 1;
		K.colOrigin = 1;
		break;
	case K_SQUARE_3:
		K.Rows = 3;
		K.Cols = 3;
		K.rowOrigin = 1;
		K.colOrigin = 1;
		break;
	}

	/* Init hist in L1 */
	K.kernel = (uint8_t *) pi_fc_l1_malloc((uint32_t) K.Rows*K.Cols);
	if (K.kernel == NULL)
	{
		printf("buff alloc failed !\n");
		pmsis_exit(-1);
	}
	p = K.kernel;

	switch (e){
	case K_STAR_3:
		//{0, 1, 0}
		//{1, 1, 1}
		//{0, 1, 0}
		*p++ = 0; *p++ = 1; *p++ = 0;
		*p++ = 1; *p++ = 1; *p++ = 1;
		*p++ = 0; *p++ = 1; *p++ = 0;
		break;
	case K_CIRCLE_3:
		//{0, 1, 0}
		//{1, 1, 1}
		//{0, 1, 0}
		*p++ = 0; *p++ = 1; *p++ = 0;
		*p++ = 1; *p++ = 1; *p++ = 1;
		*p++ = 0; *p++ = 1; *p++ = 0;
		break;
	case K_SQUARE_3:
		//{1, 1, 1}
		//{1, 1, 1}
		//{1, 1, 1}
		*p++ = 1; *p++ = 1; *p++ = 1;
		*p++ = 1; *p++ = 1; *p++ = 1;
		*p++ = 1; *p++ = 1; *p++ = 1;
		break;
	}


#if PRINT_IMG
	printf("Kernel\n");
	for(int RR = 0; RR < (K.Rows*K.Cols); RR++){
		printf("%d ,",K.kernel[RR]);
	}
	printf("\n");
#endif

	return K;

}

void ClearDefaultKernel(m_kernel_t K){

	pi_fc_l1_free(K.kernel, (uint32_t) K.Rows*K.Cols);

}

//////////////////
//   EROSION     /
//////////////////
void Erosion(uint8_t *img, uint8_t *dst, int Rows, int Cols, m_kernel_t K) {

#if PRINT_IMG
	printf("Kernel Erosion\n");
	/*for(int RR = 0; RR < (K.Rows*K.Cols); RR++){
		printf("%d ,",K.kernel[RR]);
	}
	printf("\n");
	printf("K rows: %d\n",K.Rows);
	printf("K cols: %d\n",K.Cols);*/
#endif

#if PERF
	uint32_t tt;
	pi_perf_conf(1 << PI_PERF_CYCLES | 1 << PI_PERF_ACTIVE_CYCLES);
	pi_perf_start();
	tt = pi_perf_read(PI_PERF_ACTIVE_CYCLES);
#endif

	/* Init hist in L1 */
	/*uint8_t * in = (uint8_t *) pi_fc_l1_malloc((uint32_t) K.Rows*Cols);
	if (in == NULL)
	{
		printf("buff alloc failed !\n");
		pmsis_exit(-1);
	}*/

	//morphFrameZERO();
	int RR = K.Rows - K.rowOrigin - 1;
	int CC = K.Cols - K.colOrigin - 1;
	int Kpp = (K.rowOrigin*K.Cols)+K.colOrigin;

	for (int currentRow = 0; currentRow < Rows ; currentRow++) {

		if ((currentRow >= K.rowOrigin) && (currentRow < (Rows - RR))){

			for (int currentCol = 0; currentCol < Cols; currentCol++) {

				int pp = (currentRow*Cols)+currentCol;
				if ((currentCol >= K.colOrigin) && (currentCol < (Cols - CC))){

					if (img[pp]) {

						////////////////////
						// Test for Erosion /
						////////////////////

						int rowStart = (currentRow - K.rowOrigin);
						int rowEnd = (rowStart + K.Rows);
						int colStart = (currentCol - K.colOrigin);
						int colEnd = (colStart + K.Cols);

						int k_c = 0;
						uint8_t flag = 1;

						for (int i = rowStart; i < rowEnd; i++) {
							for (int j = colStart; j < colEnd; j++) {

								/* since the kernel is raw*cols it can proceed straightforward */
								if (K.kernel[k_c]) {
									/* in */
									if (!img[(i*Cols)+j]) {
										flag = 0;
										goto escape;
									}
								}
								k_c++;

							}
						}
						/* goto */
						escape: dst[pp] = flag;
					}else{
						dst[pp] = 0;
					}
				}else{
					dst[pp] = 0;
				}
			}// for cols
		}else{
			memset(&dst[(currentRow*Cols)],0,Cols);
		}// if rows
	}//for rows

#if PERF
	pi_perf_stop();
	//printf("Img Size: %d \n",(int)(Rows*Cols));
	printf("Cycles %d \r\n",(int)(pi_perf_read(PI_PERF_ACTIVE_CYCLES)-tt));
#endif

#if PRINT_IMG
	for(RR = 0; RR < PRINT_IMG; RR++){
		printf("%d ,",dst[RR]);
	}
	printf("\r\n");
#endif

}


void Dilation(uint8_t *img, uint8_t *dst, int Rows, int Cols, m_kernel_t K){

#if PRINT_IMG
	printf("Kernel Erosion\r\n");
	/*for(int RR = 0; RR < (K.Rows*K.Cols); RR++){
		printf("%d ,",K.kernel[RR]);
	}
	printf("\n");
	printf("K rows: %d\n",K.Rows);
	printf("K cols: %d\n",K.Cols);*/
#endif

#if PERF
	uint32_t tt;
	pi_perf_conf(1 << PI_PERF_CYCLES | 1 << PI_PERF_ACTIVE_CYCLES);
	pi_perf_start();
	tt = pi_perf_read(PI_PERF_ACTIVE_CYCLES);
#endif

	//morphFrameZERO();
	int RR = K.Rows - K.rowOrigin - 1;
	int CC = K.Cols - K.colOrigin - 1;
	int Kpp = (K.rowOrigin*K.Cols)+K.colOrigin;

	for (int currentRow = K.rowOrigin; currentRow < (Rows - RR) ; currentRow++) {
		for (int currentCol = K.colOrigin; currentCol < (Cols - CC); currentCol++) {

			int pp = (currentRow*Cols)+currentCol;

			if (img[pp]) {

				////////////////////
				// Test for Dilation /
				////////////////////

				int rowStart = (currentRow - K.rowOrigin);
				int rowEnd = (rowStart + K.Rows);
				int colStart = (currentCol - K.colOrigin);
				int colEnd = (colStart + K.Cols);

				int k_c = 0;

				for (int i = rowStart; i < rowEnd; i++) {
					for (int j = colStart; j < colEnd; j++) {

						if (K.kernel[k_c]) {
							dst[(i*Cols)+j] = K.kernel[k_c];
						}else{
							dst[(i*Cols)+j] = img[(i*Cols)+j];
						}
						k_c++;
					}
				}
			}else{
				dst[pp] = 0;
			}

		}
	}

#if PERF
	pi_perf_stop();
	//printf("Img Size: %d \n",(int)(Rows*Cols));
	printf("Cycles %d \r\n",(int)(pi_perf_read(PI_PERF_ACTIVE_CYCLES)-tt));
#endif

#if PRINT_IMG
	for(RR = 0; RR < PRINT_IMG; RR++){
		printf("%d ,",dst[RR]);
	}
	printf("\r\n");
#endif

}

void Opening( uint8_t *img, int Rows, int Cols, m_kernel_t K ) {

	uint8_t * buff = (uint8_t *) pmsis_l2_malloc((uint32_t) Rows*Cols);
	if (buff == NULL)
	{
		printf("buff alloc failed !\r\n");
		pmsis_exit(-1);
	}

	Erosion(img, buff, Rows, Cols, K);
	Dilation(buff, img, Rows, Cols, K);

	pmsis_l2_malloc_free(buff, (uint32_t) Rows*Cols);

}

void Closing( uint8_t *img, int Rows, int Cols, m_kernel_t K ) {

	uint8_t * buff = (uint8_t *) pmsis_l2_malloc((uint32_t) Rows*Cols);
	if (buff == NULL)
	{
		printf("buff alloc failed !\r\n");
		pmsis_exit(-1);
	}

	Dilation(img, buff, Rows, Cols, K);
	Erosion(buff, img, Rows, Cols, K);

	pmsis_l2_malloc_free(buff, (uint32_t) Rows*Cols);

}


int Img_Ones_Count (uint8_t *img, int size) {

	int errors = 0;
    int *po = (int *)img;
    int *pe = (int *)img + (size/4);
    while (po < pe){
    	int a = *po++;
    	int b = *po++;
    	errors += __builtin_pulp_cnt(a);
    	errors += __builtin_pulp_cnt(b);
    }

    return errors;

}




